// File   : com.fredswartz.ezgui/EZPanel.java
// Purpose: Provide an easy interface to GridBagLayout.
// Author : Fred Swartz
// Date   : 2005-02-21
// Version: 0.5

package com.fredswartz.ezgui;

import java.awt.*;
import javax.swing.*;
import javax.swing.border.*;
import java.util.*;

// Bugs:
//    col() is certainly not working.
//
// Enhancements:
//    Movement: rowCol(r, c), left(), ...
//    addPair(string, comp) so vertical adds work for common case?
//    The list of autoExpand components must be expanded, but 
//             more importantly, it must be array driven and under
//             user control.
//
// Questions:
//    ? Replace alignment methods.  north() -> align("N"), ...
//    ? Use shorter names, eg, w(2) for width(2)?
//

/**
 * EZPanel (GridBagPanel) is a subclass of JPanel that uses 
 * GridBaglayout and provides a number of convenience methods
 * for laying out components.
 * @author Fred Swartz
 * @version 2005-02-11
 */
public class EZPanel extends JPanel {
    
    //... Gaps - The default gap size of 5 generally looks good and is
    //    largely consistent with the Human Interface Guidelines.
    //    The default gap size can be changed by setGap(),
    //    (conflicting gap sizes with default to the largest).
    //    A single GridBagConstraints object is used for all gap constraints.
    public static final int DEFAULT_BORDER_SIZE = 12;
    public static final int DEFAULT_GAP_SIZE = 5;
    private GridBagConstraints m_gapGbc = new GridBagConstraints();
    private int m_gapSize = DEFAULT_GAP_SIZE;
    
    /** Controls direction of default movement when adding.
     *  The value is set true following a newRow() call, and false
     *  following a newCol() call.
     *  The default is to accross rows.
     */
    enum Direction {RIGHT, DOWN, UNKNOWN}
    
    private Direction m_direction = Direction.UNKNOWN;
    
    /** This is set true when there is no need to automatically move to a new
     *  position.  Both the attribute methods and add() method will move the 
     *  cursor if this is false, otherwise they s.
     */
    private boolean m_isNewPosition  = true;
    
    /* If true checks for certain components and may set horizontal and 
     * vertical expansion to true.  Eg, JTextFields can be horizontally 
     * expanded and JTextAreas can be both horizontally and vertically 
     * expanded.  Needs to be augmented with array of such components.
     * Can be changed with the setAutoExpand(t/f) user method.
     */
    private boolean m_autoExpand = true;
        
    /** The single constraints obj for adding all components. 
     *  It functions like a cursor as most methods make movements
     *  relative to its current value.
     */
    private GridBagConstraints m_cursor = new GridBagConstraints();
    
    /** Used to check for components added more than once. */
    private Set<Component> m_duplicateComponentCheck = new HashSet<Component>();
    
    /** Used to check for overlapping component areas. */
    private Set<Integer>  m_areaOverlapCheck = new HashSet<Integer>();
    
    /** Outputs info on System.out an where each component, gap is added.
     *  Prints internal GridBagConstraints coordinates.
     *  Can be set on/off with the user setTrace() method.
     */
    private boolean m_trace = false;
    
    /* m_nextRowGap and m_nextColGap record where next gap would be
     * generated.
     */
    private int m_nextRowGap = 2;
    private int m_nextColGap = 2;
    
    //====================================================== constructor default
    public EZPanel() {  
        this(c_emptyBorder(DEFAULT_BORDER_SIZE));
    }
    
    //============================================================== constructor
    public EZPanel(int borderSize) {
        this(c_emptyBorder(borderSize));
    }
    
    //============================================================== constructor
    public EZPanel(String borderTitle) {
        this(c_titledBorder(borderTitle));
    }
    
    //============================================================== constructor
    public EZPanel(Border brdr) {
        border(brdr);  // Set default border.  Can be changed with border().
        
        //... Set GridBagConstraints parameters.
        //    Row 0 and Col 0 are used for adding gap struts.
        m_cursor.gridx = 1;
        m_cursor.gridy = 1;
        m_reset();
        
        //... Use an underlying GridBagLayout
        this.setLayout(new GridBagLayout());
    }
    
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    // Attribute methods
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    
    //================================================================ north etc.
    public EZPanel north()     { return m_align(GridBagConstraints.NORTH);     }
    public EZPanel northeast() { return m_align(GridBagConstraints.NORTHEAST); }
    public EZPanel northwest() { return m_align(GridBagConstraints.NORTHWEST); }
    public EZPanel south()     { return m_align(GridBagConstraints.SOUTH);     }
    public EZPanel southeast() { return m_align(GridBagConstraints.SOUTHEAST); }
    public EZPanel southwest() { return m_align(GridBagConstraints.SOUTHWEST); }
    public EZPanel east()      { return m_align(GridBagConstraints.EAST);      }
    public EZPanel west()      { return m_align(GridBagConstraints.WEST);      }
    public EZPanel center()    { return m_align(GridBagConstraints.CENTER);    }
    
    //================================================================= widthRem
    /** Sets the cursor width to remainder of row/col. */
    public EZPanel widthRem() {
        m_defaultMove();
        m_cursor.gridwidth = GridBagConstraints.REMAINDER;
        return this;
    }
    
    //================================================================ heightRem
    /** Sets the cursor height to remainder of row/col. */
    public EZPanel heightRem() {
        m_defaultMove();
        m_cursor.gridheight = GridBagConstraints.REMAINDER;
        return this;
    }
    
    //==================================================================== width
    /** Sets the cursor width to "width". */
    public EZPanel width(int width) {
        assert width > 0 : "EZPanel: width must be > 0";
        m_defaultMove();
        m_cursor.gridwidth = width*2 - 1;
        return this;
    }
    
    //=================================================================== height
    /** Sets the cursor width to "height". */
    public EZPanel height(int height) {
        assert height > 0 : "EZPanel: height must be > 0";
        m_defaultMove();
        m_cursor.gridheight = height*2 - 1;
        return this;
    }
    
    //================================================================== expandH
    /** Sets cursor to expand area horizontally. */
    public EZPanel expandH() {
        m_defaultMove();
        m_cursor.weightx = 1.0;
        return this;
    }
    
    //================================================================== expandV
    /** Sets cursor to expand area vertically. */
    public EZPanel expandV() {
        m_defaultMove();
        m_cursor.weighty = 1.0;
        return this;
    }
    
    //================================================================= expandHV
    /** Sets cursor to expand area horizontally and vertically. */
    public EZPanel expandHV() {
        expandH();
        expandV();
        return this;
    }
    
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    // Movement methods
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    
    //====================================================================== row
    public EZPanel row() {
        if (m_direction != Direction.UNKNOWN) {
            //... Move down only if not just starting (ie, UNKNOWN).
            m_cursor.gridy += 2;
        }
        row(m_cursor.gridy/2 + 1);  // Translate here to external coords.
        return this;
    }
    
    //====================================================================== row
    public EZPanel row(int r) {
        m_cursor.gridx = 1;
        m_cursor.gridy = 2*r - 1;      // Translate to internal coords
        m_direction = Direction.RIGHT; // Default additions to right.
        m_reset();
        return this;
    }
    
    //====================================================================== col
    public EZPanel col() {
        //??? Needs to be fixed.
        m_right();
        m_cursor.gridy = 1;
        m_direction = Direction.DOWN; // Default additions down.
        return this;
    }
    
    //===================================================================== skip
    // Possible alternatives: right(), next(), add()
    public EZPanel skip() {
        m_isNewPosition = false;  // Make it possible to move.
        m_defaultMove();
        return this;
    }
    
    //===================================================================== down
    public EZPanel down() {
        m_down();
        return this;
    }
    
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    // Add components
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    
    //====================================================================== add
    /** Adds a component at the current position, moving by default
     *  to the right if necessary.
     * @param comp A GUI compontent
     * @return this for call chaining.
     */
    public EZPanel add(Component comp) {
        m_defaultRight();
        m_defaultExpansion(comp);
        m_add(comp, m_cursor);
        m_isNewPosition = false;
        return this;
    }
    
    //====================================================================== add
    public EZPanel add(String labelText) {
        add(new JLabel(labelText));
        return this;
    }
    
    //=============================================================== addHSpring
    public EZPanel addHSpring() {
        return expandH().add(Box.createHorizontalGlue());
    }
    
    //=============================================================== addVSpring
    public EZPanel addVSpring() {
        return expandV().add(Box.createVerticalGlue());
    }
    
    //================================================================ addHSpace
    public EZPanel addHSpace(int pixels) {
        return add(Box.createHorizontalStrut(pixels));
    }
    
    //================================================================ addVSpace
    public EZPanel addVSpace(int pixels) {
        return add(Box.createVerticalStrut(pixels));
    }
    
    //================================================================= addHLine
    public EZPanel addHLine() {
        return add(new JSeparator(SwingConstants.HORIZONTAL));
    }
    
    //================================================================= addVLine
    public EZPanel addVLine() {
        return add(new JSeparator(SwingConstants.VERTICAL));
    }
    
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    // Border methods
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    
    //=================================================================== border
    public EZPanel border(Border border) {
        this.setBorder(border);
        return this;
    }
    
    //=================================================================== border
    public EZPanel border(int borderSize) {
        this.setBorder(c_emptyBorder(borderSize));
        return this;
    }
    
    //=================================================================== border
    public EZPanel border(String borderTitle) {
        this.setBorder(c_titledBorder(borderTitle));
        return this;
    }
    
    //=================================================================== setGap
    public EZPanel setGap(int gapSize) {
        m_gapSize = gapSize;
        return this;
    }
    
    //=============================================================== autoExpand
    public EZPanel autoExpand(boolean autoExpand) {
        m_autoExpand = autoExpand;
        return this;
    }
    
    //=============================================================== autoExpand
    public EZPanel autoExpand() {
        autoExpand(true);
        return this;
    }
    
    //================================================================= trace
    public EZPanel trace(boolean trace) {
        m_trace = trace;
        return this;
    }
    
    //================================================================= trace
    public EZPanel trace() {
        trace(true);
        return this;
    }
    
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    // Private utility methods
    //\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\
    
    //================================================================== m_align
    private EZPanel m_align(int anchorValue) {
        m_defaultMove();
        m_cursor.fill   = GridBagConstraints.NONE;
        m_cursor.anchor = anchorValue;
        return this;
    }
    
    //======================================================= m_defaultExpansion
    private void m_defaultExpansion(Component comp) {
        if (m_autoExpand) {
            boolean expandx = (comp instanceof JTextField)
            || (comp instanceof JTextArea)
            || (comp instanceof JScrollPane);
            
            boolean expandy = (comp instanceof JTextArea)
            || (comp instanceof JScrollPane);
            
            if (expandx) m_cursor.weightx = 1.0;
            if (expandy) m_cursor.weighty = 1.0;
        }
    }
    
    
    //============================================================ m_defaultMove
    /** Move either right or down */
    private void m_defaultMove() {
        switch (m_direction) {
            case RIGHT:  m_defaultRight(); break;
            case DOWN :  m_defaultDown();  break;
            default: throw new RuntimeException("EZPanel: No direction.");
        }
    }
    
    //=========================================================== m_defaultRight
    private void m_defaultRight() {
        if (!m_isNewPosition) {
            m_right();
        }
    }
    
    //================================================================== m_right
    private void m_right() {
        if (m_cursor.gridwidth == GridBagConstraints.REMAINDER) {
            throw new RuntimeException("EZPanel: Can't move right past end");
        }
        
        //... Move over gap to next cell.
        m_cursor.gridx += m_cursor.gridwidth + 1;  // Move right
        
        //... Reset all gbc params except row and col.
        m_reset();
    }
    
    //============================================================ m_defaultDown
    private void m_defaultDown() {
        if (!m_isNewPosition) {
            m_down();
        }
    }
    
    //=================================================================== m_down
    /* Move down by height of last component area. */
    private void m_down() {
        if (m_cursor.gridheight == GridBagConstraints.REMAINDER) {
            throw new RuntimeException("EZPanel: Can't move beyond bottom");
        }
        
        //... Move over gap to next cell.
        m_cursor.gridy += m_cursor.gridheight + 1;  // Move down
        
        //... Reset all gbc params except row and col.
        m_reset();
    }
    
    //================================================================== m_reset
    private void m_reset() {
        m_cursor.gridwidth  = 1;
        m_cursor.gridheight = 1;
        m_cursor.fill    = GridBagConstraints.BOTH;
        m_cursor.weightx = 0.0;
        m_cursor.weighty = 0.0;
        
        m_isNewPosition = true;
    }
    
    //==================================================================== m_add
    /** All non-gap adds to the layout are untimately funneled thru here.
     *  The purpose is to provide a single point where checking
     *  for overapping components can be performed.
     */
    private void m_add(Component comp, GridBagConstraints gbc) {
        //... Trace
        if (m_trace) {
            //... Map internal to external coords.  Gaps might print strangely
            System.out.print("trace: " + comp.getClass().getName());
            System.out.println(" added at row=" + (gbc.gridy+1)/2
                                      + " col=" + (gbc.gridx+1)/2
                                      + ((gbc.gridwidth  > 1) ? " w=" + (gbc.gridwidth+1)/2 : "")
                                      + ((gbc.gridheight > 1) ? " h=" + (gbc.gridheight+1)/2: ""));
        }
        
        //... Check for same component added more than once.
        if (!m_duplicateComponentCheck.add(comp)) {
            throw new RuntimeException("EZPanel: Component added twice to layout: "
                    + comp.getClass().getName());
        }
        
        //... Check for (illegal) overlaps.  Compute unique int for row, col combination.
        //    Redundant gap additions are ignored.
        for (int row=gbc.gridy; row < gbc.gridy+gbc.gridheight; row++) {
            for (int col=gbc.gridx; col < gbc.gridx+gbc.gridwidth; col++) {
                if (!m_areaOverlapCheck.add(row * 1000 + col)) {
                    if (row==0 || col==0) {
                        //... This is a gap filler and there's already one there,
                        //    so simply ignore this redundant add.
                        return; //>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
                    } else {
                        throw new RuntimeException("EZPanel: Component overlaps previous component: "
                            + comp.getClass().getName());
                    }
                }
            }
        }
        
        //... Add it to the real GridBagLayout
        super.add(comp, gbc);
        
        //... Add row gaps if necessary
        int possibleLastRowGap = gbc.gridy + Math.max(gbc.gridheight, 1);
        // System.out.println("       m_nextRowGap=" + m_nextRowGap + ", possibleLastRowGap=" + possibleLastRowGap);
        while (m_nextRowGap < possibleLastRowGap) {
            //... Add a vertical gap
            m_addRowGap(m_nextRowGap);
            m_nextRowGap += 2;
        }
        
        //... Add col gaps if necessary
        int possibleLastColGap = gbc.gridx + Math.max(gbc.gridwidth, 1);
        // System.out.println("       m_nextColGap=" + m_nextColGap + ", possibleLastColGap=" + possibleLastColGap);
        while (m_nextColGap < possibleLastColGap) {
            //... Add a horizontal gap
            m_addColGap(m_nextColGap);
            m_nextColGap += 2;
        }
    }
    
    //============================================================== m_addRowGap
    private void m_addRowGap(int internalRow) {
        if (m_gapSize > 0) {
            if (m_trace) System.out.println("       Add row gap at " + internalRow);
            m_gapGbc.gridx = 0;
            m_gapGbc.gridy = internalRow;
            add(Box.createVerticalStrut(m_gapSize), m_gapGbc);
        }
    }
    
    //============================================================== m_addColGap
    private void m_addColGap(int internalCol) {
        if (m_gapSize > 0) {
            if (m_trace) System.out.println("       Add col gap at " + internalCol);
            m_gapGbc.gridy = 0;
            m_gapGbc.gridx = internalCol;
            add(Box.createHorizontalStrut(m_gapSize), m_gapGbc);
        }
    }
    
    //============================================================ m_emptyBorder
    private static Border c_emptyBorder(int bSize) {
        return BorderFactory.createEmptyBorder(bSize, bSize, bSize, bSize);
    }
    
    //============================================================ m_emptyBorder
    private static Border c_titledBorder(String title) {
        Border etchedBdr  = BorderFactory.createEtchedBorder();
        Border titledBdr  = BorderFactory.createTitledBorder(etchedBdr, title);
        Border emptyBdr   = BorderFactory.createEmptyBorder(5,5,5,5);
        return BorderFactory.createCompoundBorder(titledBdr, emptyBdr);
    }
}